<?php
require_once __DIR__ . "/../../config/auth.php";
require_once __DIR__ . "/../../config/db.php";
require_once __DIR__ . "/../../includes/helpers.php";

$id = (int)($_GET['id'] ?? 0);
if (!$id) {
    flash_set('error', 'Missing batch id.');
    redirect('batches_list.php');
}

// 1) Load batch
$stmt = $conn->prepare("SELECT * FROM production_batches WHERE id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();
$batch = $stmt->get_result()->fetch_assoc();

if (!$batch) {
    flash_set('error', 'Batch not found.');
    redirect('batches_list.php');
}

if ($batch['status'] === 'Posted') {
    flash_set('error', 'Batch already posted.');
    redirect('batches_list.php');
}

$product_id = (int)$batch['product_id'];
$actual_qty = (int)$batch['actual_qty'];

// 2) Load recipe for this product
$sql = "SELECT r.id, r.yield_qty 
        FROM recipes r 
        WHERE r.product_id = ? 
        ORDER BY r.id DESC 
        LIMIT 1";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $product_id);
$stmt->execute();
$recipe = $stmt->get_result()->fetch_assoc();

if (!$recipe) {
    flash_set('error', 'No recipe found for this product. Set up recipe first.');
    redirect('batches_list.php');
}

$recipe_id = (int)$recipe['id'];
$yield_qty = max(1, (int)$recipe['yield_qty']);

// Number of recipe batches achieved (based on actual output)
$factor = $actual_qty / $yield_qty;
if ($factor <= 0) {
    flash_set('error', 'Actual quantity must be greater than zero.');
    redirect('batches_list.php');
}

// Use transaction so everything moves together
$conn->begin_transaction();

try {
    $now = date('Y-m-d H:i:s');

    // 3) Consume raw materials according to recipe_items * factor
    $sql = "SELECT ri.material_id, ri.qty, rm.current_stock 
            FROM recipe_items ri 
            JOIN raw_materials rm ON ri.material_id = rm.id
            WHERE ri.recipe_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $recipe_id);
    $stmt->execute();
    $resItems = $stmt->get_result();

    while($it = $resItems->fetch_assoc()) {
        $material_id = (int)$it['material_id'];
        $qty_per_batch = (float)$it['qty'];
        $qty_issue = $qty_per_batch * $factor; // how much to consume

        // insert raw_material_moves
        $stmt2 = $conn->prepare("INSERT INTO raw_material_moves 
            (move_date, material_id, qty, direction, ref_type, ref_id, remarks)
            VALUES (?,?,?,?,?,?,?)");
        $direction = -1;
        $ref_type = 'PRODUCTION';
        $remarks = 'Auto issue from batch ' . $batch['batch_no'];
        $stmt2->bind_param("sdiisis", $now, $material_id, $qty_issue, $direction, $ref_type, $id, $remarks);
        $stmt2->execute();

        // update raw_materials.current_stock
        $stmt3 = $conn->prepare("UPDATE raw_materials 
                                 SET current_stock = current_stock - ? 
                                 WHERE id = ?");
        $stmt3->bind_param("di", $qty_issue, $material_id);
        $stmt3->execute();
    }

    // 4) Add finished goods to stock (actual_qty)
    // upsert into finished_goods_stock
    $stmt = $conn->prepare("SELECT id, on_hand FROM finished_goods_stock WHERE product_id = ?");
    $stmt->bind_param("i", $product_id);
    $stmt->execute();
    $fg = $stmt->get_result()->fetch_assoc();

    if ($fg) {
        $stmt = $conn->prepare("UPDATE finished_goods_stock 
                                SET on_hand = on_hand + ? 
                                WHERE product_id = ?");
        $stmt->bind_param("ii", $actual_qty, $product_id);
        $stmt->execute();
    } else {
        $stmt = $conn->prepare("INSERT INTO finished_goods_stock (product_id, on_hand) VALUES (?,?)");
        $stmt->bind_param("ii", $product_id, $actual_qty);
        $stmt->execute();
    }

    // log finished_goods_moves
    $stmt = $conn->prepare("INSERT INTO finished_goods_moves 
        (move_date, product_id, qty, direction, ref_type, ref_id, remarks)
        VALUES (?,?,?,?,?,?,?)");
    $direction = +1;
    $ref_type = 'PRODUCTION';
    $remarks = 'Batch ' . $batch['batch_no'];
    $stmt->bind_param("siiisis", $now, $product_id, $actual_qty, $direction, $ref_type, $id, $remarks);
    $stmt->execute();

    // 5) Mark batch as Posted
    $stmt = $conn->prepare("UPDATE production_batches 
                            SET status = 'Posted', posted_at = ? 
                            WHERE id = ?");
    $stmt->bind_param("si", $now, $id);
    $stmt->execute();

    $conn->commit();
    flash_set('success', 'Batch posted to stock successfully.');
} catch (Exception $e) {
    $conn->rollback();
    flash_set('error', 'Error posting batch: ' . $e->getMessage());
}

redirect('batches_list.php');
