<?php
require_once __DIR__ . "/../../config/auth.php";
require_once __DIR__ . "/../../config/db.php";
require_once __DIR__ . "/../../includes/helpers.php";
csrf_check();

$mode          = $_POST['mode'] ?? 'normal';
$sale_date     = trim($_POST['sale_date'] ?? date('Y-m-d H:i:s'));
$receipt_no    = trim($_POST['receipt_no'] ?? '');
$customer_name = trim($_POST['customer_name'] ?? '');
$payment_mode  = trim($_POST['payment_mode'] ?? 'Cash');
$cash_amount   = (float)($_POST['cash_amount'] ?? 0);
$mpesa_amount  = (float)($_POST['mpesa_amount'] ?? 0);
$cake_order_id = !empty($_POST['cake_order_id']) ? (int)$_POST['cake_order_id'] : null;

if (!$receipt_no) {
    flash_set('error', 'Receipt number is required.');
    redirect('counter.php');
}

// Items
$product_ids = $_POST['product_id'] ?? [];
$qtys        = $_POST['qty'] ?? [];
$prices      = $_POST['unit_price'] ?? [];

$lines = [];
$total_items_amount = 0;

for ($i = 0; $i < count($product_ids); $i++) {
    $pid = (int)$product_ids[$i];
    $q   = (int)$qtys[$i];
    $up  = (float)$prices[$i];

    if ($pid && $q > 0 && $up > 0) {
        $lt = $q * $up;
        $lines[] = [
            'product_id' => $pid,
            'qty'        => $q,
            'unit_price' => $up,
            'line_total' => $lt,
        ];
        $total_items_amount += $lt;
    }
}

$payment_total = $cash_amount + $mpesa_amount;
$created_by    = $_SESSION['user_id'] ?? null;
$hasItems      = !empty($lines);

if (!$hasItems && !$cake_order_id) {
    flash_set('error', 'Add at least one item OR select a cake order to pay.');
    redirect('counter.php');
}
if ($payment_total <= 0) {
    flash_set('error', 'Payment must be greater than zero.');
    redirect('counter.php');
}

$conn->begin_transaction();

try {
    $pos_id = null;

    // ---- CASE A: NORMAL ITEM SALE (with or without linking to cake) ----
    if ($hasItems) {
        // 1) Stock check
        foreach ($lines as $ln) {
            $pid = $ln['product_id'];
            $qty = $ln['qty'];

            $stmt = $conn->prepare("
              SELECT on_hand 
              FROM finished_goods_stock 
              WHERE product_id = ? 
              FOR UPDATE
            ");
            $stmt->bind_param("i", $pid);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $on_hand = $row ? (int)$row['on_hand'] : 0;

            if ($on_hand < $qty) {
                $conn->rollback();
                flash_set('error', "Insufficient stock for one of the items (product ID {$pid}). On hand: {$on_hand}, needed: {$qty}.");
                redirect('counter.php');
            }
        }

        // 2) Insert POS header
        if ($cake_order_id) {
            $stmt = $conn->prepare("
              INSERT INTO pos_sales
              (sale_date, cake_order_id, receipt_no, customer_name, payment_mode,
               cash_amount, mpesa_amount, total_amount, created_by)
              VALUES (?,?,?,?,?,?,?,?,?)
            ");
            $stmt->bind_param(
                "sisssdddi",
                $sale_date,
                $cake_order_id,
                $receipt_no,
                $customer_name,
                $payment_mode,
                $cash_amount,
                $mpesa_amount,
                $total_items_amount,
                $created_by
            );
        } else {
            $stmt = $conn->prepare("
              INSERT INTO pos_sales
              (sale_date, receipt_no, customer_name, payment_mode,
               cash_amount, mpesa_amount, total_amount, created_by)
              VALUES (?,?,?,?,?,?,?,?)
            ");
            $stmt->bind_param(
                "ssssdddi",
                $sale_date,
                $receipt_no,
                $customer_name,
                $payment_mode,
                $cash_amount,
                $mpesa_amount,
                $total_items_amount,
                $created_by
            );
        }
        $stmt->execute();
        $pos_id = $conn->insert_id;

        // 3) Insert items + deduct stock + log moves
        $stmtItem = $conn->prepare("
          INSERT INTO pos_sale_items
          (pos_sale_id, product_id, qty, unit_price, line_total)
          VALUES (?,?,?,?,?)
        ");

        $stmtUpdStock = $conn->prepare("
          UPDATE finished_goods_stock
          SET on_hand = on_hand - ?
          WHERE product_id = ?
        ");

        $stmtMove = $conn->prepare("
          INSERT INTO finished_goods_moves
          (move_date, product_id, qty, direction, ref_type, ref_id, remarks)
          VALUES (?,?,?,?,?,?,?)
        ");

        $now = date('Y-m-d H:i:s');
        foreach ($lines as $ln) {
            $pid = $ln['product_id'];
            $qty = $ln['qty'];
            $up  = $ln['unit_price'];
            $lt  = $ln['line_total'];

            // Items table
            $stmtItem->bind_param("iiidd", $pos_id, $pid, $qty, $up, $lt);
            $stmtItem->execute();

            // Deduct stock
            $stmtUpdStock->bind_param("ii", $qty, $pid);
            $stmtUpdStock->execute();

            // Movement log
            $direction = -1;
            $ref_type  = 'POS';
            $remarks   = 'Counter sale ' . $receipt_no;
            $stmtMove->bind_param("siiisis", $now, $pid, $qty, $direction, $ref_type, $pos_id, $remarks);
            $stmtMove->execute();
        }

    // ---- CASE B: CAKE PAYMENT ONLY (no items, just money against cake order) ----
    } else {
        // Validate cake order exists
        $stmt = $conn->prepare("
          SELECT total_amount
          FROM cake_orders
          WHERE id = ?
          FOR UPDATE
        ");
        $stmt->bind_param("i", $cake_order_id);
        $stmt->execute();
        $order = $stmt->get_result()->fetch_assoc();

        if (!$order) {
            $conn->rollback();
            flash_set('error', 'Cake order not found.');
            redirect('counter.php');
        }

        $total_order = (float)$order['total_amount'];

        // Compute already paid
        $stmt = $conn->prepare("
          SELECT IFNULL(SUM(cash_amount + mpesa_amount),0) AS paid
          FROM pos_sales
          WHERE cake_order_id = ?
        ");
        $stmt->bind_param("i", $cake_order_id);
        $stmt->execute();
        $agg = $stmt->get_result()->fetch_assoc();
        $paid_so_far = (float)($agg['paid'] ?? 0);
        $current_balance = max(0, $total_order - $paid_so_far);

        if ($payment_total > $current_balance + 0.01) {
            $conn->rollback();
            flash_set('error', 'Payment cannot be more than current cake balance.');
            redirect('counter.php');
        }

        // Insert POS header as payment-only (no items, no stock)
        $stmt = $conn->prepare("
          INSERT INTO pos_sales
          (sale_date, cake_order_id, receipt_no, customer_name, payment_mode,
           cash_amount, mpesa_amount, total_amount, created_by)
          VALUES (?,?,?,?,?,?,?,?,?)
        ");
        $stmt->bind_param(
            "sisssdddi",
            $sale_date,
            $cake_order_id,
            $receipt_no,
            $customer_name,
            $payment_mode,
            $cash_amount,
            $mpesa_amount,
            $payment_total,
            $created_by
        );
        $stmt->execute();
        $pos_id = $conn->insert_id;
    }

    // ---- UPDATE CAKE ORDER PAYMENTS (for both cases if cake_order_id present) ----
    if ($cake_order_id) {
        // Get total order again (lock row)
        $stmt = $conn->prepare("
          SELECT total_amount, status
          FROM cake_orders
          WHERE id = ?
          FOR UPDATE
        ");
        $stmt->bind_param("i", $cake_order_id);
        $stmt->execute();
        $order = $stmt->get_result()->fetch_assoc();

        if ($order) {
            $total_order = (float)$order['total_amount'];

            // Sum all POS payments for this cake order
            $stmt = $conn->prepare("
              SELECT IFNULL(SUM(cash_amount + mpesa_amount),0) AS paid
              FROM pos_sales
              WHERE cake_order_id = ?
            ");
            $stmt->bind_param("i", $cake_order_id);
            $stmt->execute();
            $agg = $stmt->get_result()->fetch_assoc();
            $paid_total = (float)($agg['paid'] ?? 0);

            $balance = max(0, $total_order - $paid_total);

            if ($paid_total <= 0) {
                $payment_status = 'Pending';
            } elseif ($balance > 0) {
                $payment_status = 'Partially Paid';
            } else {
                $payment_status = 'Paid';
            }

            // auto move Enquiry -> Confirmed on first payment
            $new_status = $order['status'];
            if ($order['status'] === 'Enquiry' && $paid_total > 0) {
                $new_status = 'Confirmed';
            }

            $stmt = $conn->prepare("
              UPDATE cake_orders
              SET deposit_amount = ?, balance_amount = ?, payment_status = ?, status = ?
              WHERE id = ?
            ");
            $stmt->bind_param("ddssi", $paid_total, $balance, $payment_status, $new_status, $cake_order_id);
            $stmt->execute();
        }
    }

    $conn->commit();
    flash_set('success', 'Transaction saved successfully.');
} catch (Exception $e) {
    $conn->rollback();
    flash_set('error', 'Error saving POS transaction: ' . $e->getMessage());
}

redirect('counter.php');
