<?php
require_once __DIR__ . "/../../config/auth.php";
require_once __DIR__ . "/../../config/db.php";
require_once __DIR__ . "/../../includes/helpers.php";
csrf_check();

$load_id = (int)($_POST['load_id'] ?? 0);
if (!$load_id) {
    flash_set('error', 'Missing loading id.');
    redirect('loadings_list.php');
}

// Load header
$stmt = $conn->prepare("SELECT * FROM dispatch_loads WHERE id = ?");
$stmt->bind_param("i", $load_id);
$stmt->execute();
$load = $stmt->get_result()->fetch_assoc();

if (!$load) {
    flash_set('error', 'Loading not found.');
    redirect('loadings_list.php');
}

if ($load['status'] !== 'Posted') {
    flash_set('error', 'Only posted loadings can be reconciled.');
    redirect('loadings_list.php');
}

// Read form arrays
$dli_ids       = $_POST['dli_id'] ?? [];
$qty_returneds = $_POST['qty_returned'] ?? [];

if (empty($dli_ids) || empty($qty_returneds)) {
    flash_set('error', 'No items to update.');
    redirect('returns_form.php?id='.$load_id);
}

$conn->begin_transaction();

try {
    $now = date('Y-m-d H:i:s');

    // 1) Load mapping of item -> product_id & loaded qty
    $sql = "
      SELECT dli.id, dli.product_id, dli.qty_loaded, dli.qty_returned
      FROM dispatch_load_items dli
      WHERE dli.load_id = ?
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $load_id);
    $stmt->execute();
    $res = $stmt->get_result();
    $itemMap = [];
    while ($r = $res->fetch_assoc()) {
        $itemMap[$r['id']] = $r; // id, product_id, qty_loaded, qty_returned (previous)
    }

    // 2) Sum returns per product (we assume first-time reconciliation: previous returns were 0)
    $returnsPerProduct = [];

    for ($i = 0; $i < count($dli_ids); $i++) {
        $id  = (int)$dli_ids[$i];
        $ret = (int)$qty_returneds[$i];

        if (!isset($itemMap[$id])) {
            continue;
        }
        $loaded = (int)$itemMap[$id]['qty_loaded'];
        if ($ret < 0) $ret = 0;
        if ($ret > $loaded) $ret = $loaded; // clamp to loaded

        // Update dispatch_load_items.qty_returned
        $stmtUp = $conn->prepare("UPDATE dispatch_load_items SET qty_returned = ? WHERE id = ?");
        $stmtUp->bind_param("ii", $ret, $id);
        $stmtUp->execute();

        $product_id = (int)$itemMap[$id]['product_id'];
        if (!isset($returnsPerProduct[$product_id])) {
            $returnsPerProduct[$product_id] = 0;
        }
        $returnsPerProduct[$product_id] += $ret;
    }

    // 3) Return goods to finished_goods_stock & log moves
    foreach ($returnsPerProduct as $product_id => $qty_ret) {
        if ($qty_ret <= 0) continue;

        // Upsert finished_goods_stock
        $stmt = $conn->prepare("SELECT id, on_hand FROM finished_goods_stock WHERE product_id = ?");
        $stmt->bind_param("i", $product_id);
        $stmt->execute();
        $fg = $stmt->get_result()->fetch_assoc();

        if ($fg) {
            $stmt = $conn->prepare("
                UPDATE finished_goods_stock
                SET on_hand = on_hand + ?
                WHERE product_id = ?
            ");
            $stmt->bind_param("ii", $qty_ret, $product_id);
            $stmt->execute();
        } else {
            $stmt = $conn->prepare("
                INSERT INTO finished_goods_stock (product_id, on_hand)
                VALUES (?,?)
            ");
            $stmt->bind_param("ii", $product_id, $qty_ret);
            $stmt->execute();
        }

        // Insert finished_goods_moves
        $stmt = $conn->prepare("
            INSERT INTO finished_goods_moves
            (move_date, product_id, qty, direction, ref_type, ref_id, remarks)
            VALUES (?,?,?,?,?,?,?)
        ");
        $direction = +1; // back into bakery store
        $ref_type  = 'RETURN';
        $remarks   = 'Market returns for loading ' . $load['load_no'];
        $stmt->bind_param("siiisis", $now, $product_id, $qty_ret, $direction, $ref_type, $load_id, $remarks);
        $stmt->execute();
    }

    // 4) Mark loading as Reconciled
    $stmt = $conn->prepare("UPDATE dispatch_loads SET status = 'Reconciled' WHERE id = ?");
    $stmt->bind_param("i", $load_id);
    $stmt->execute();

    $conn->commit();
    flash_set('success', 'Returns saved, stock updated and loading reconciled.');
} catch (Exception $e) {
    $conn->rollback();
    flash_set('error', 'Error saving returns: ' . $e->getMessage());
}

redirect('returns_form.php?id='.$load_id);
