<?php
require_once __DIR__ . "/../../config/auth.php";
require_once __DIR__ . "/../../config/db.php";
require_once __DIR__ . "/../../includes/helpers.php";
csrf_token();

$load_id = (int)($_GET['id'] ?? 0);
if (!$load_id) {
    flash_set('error', 'Missing loading id.');
    redirect('loadings_list.php');
}

// Load header
$stmt = $conn->prepare("SELECT * FROM dispatch_loads WHERE id = ?");
$stmt->bind_param("i", $load_id);
$stmt->execute();
$load = $stmt->get_result()->fetch_assoc();

if (!$load) {
    flash_set('error', 'Loading not found.');
    redirect('loadings_list.php');
}

$page_title = "Returns & Reconciliation — " . $load['load_no'];

// Only allow editing when status = Posted
$editable = ($load['status'] === 'Posted');

// Load items (loaded + returned so far)
$sql = "
  SELECT dli.*, p.code, p.name, p.category
  FROM dispatch_load_items dli
  JOIN products p ON dli.product_id = p.id
  WHERE dli.load_id = ?
  ORDER BY p.category, p.name
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $load_id);
$stmt->execute();
$items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Aggregate van sales per product for this loading
$sql = "
  SELECT vsi.product_id,
         SUM(vsi.qty) AS sold_qty,
         SUM(vsi.line_total) AS sold_value
  FROM van_sales_items vsi
  JOIN van_sales vs ON vsi.van_sale_id = vs.id
  WHERE vs.load_id = ?
  GROUP BY vsi.product_id
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $load_id);
$stmt->execute();
$vsAggRes = $stmt->get_result();

$vanSalesAgg = [];
while ($r = $vsAggRes->fetch_assoc()) {
    $vanSalesAgg[$r['product_id']] = [
        'sold_qty'   => (int)$r['sold_qty'],
        'sold_value' => (float)$r['sold_value'],
    ];
}
?>
<?php require_once __DIR__ . "/../../includes/layout.php"; ?>
<div class="sb-card mb-3">
  <div class="d-flex justify-content-between align-items-center mb-2">
    <div>
      <h5 class="mb-0">Returns &amp; Reconciliation</h5>
      <p class="small text-muted mb-0">
        Load <?php echo h($load['load_no']); ?> • Date: <?php echo h($load['load_date']); ?> • Route: <?php echo h($load['route_code']); ?>
      </p>
    </div>
    <a href="loadings_list.php" class="btn btn-sm btn-outline-light rounded-pill">
      <i class="fa-solid fa-arrow-left-long me-1"></i> Back
    </a>
  </div>

  <?php if (!$editable): ?>
    <div class="alert alert-info py-2 small rounded-pill mb-3">
      This loading is already <strong><?php echo h($load['status']); ?></strong>. Returns are read-only.
    </div>
  <?php else: ?>
    <div class="alert alert-warning py-2 small rounded-pill mb-3">
      Enter market returns per product, then <strong>Save &amp; Reconcile</strong> to return unsold bread to stock
      and mark this loading as <strong>Reconciled</strong>.
    </div>
  <?php endif; ?>

  <form method="post" action="returns_save.php" class="small">
    <input type="hidden" name="load_id" value="<?php echo (int)$load_id; ?>">
    <input type="hidden" name="csrf_token" value="<?php echo h($_SESSION['csrf_token']); ?>">

    <div class="table-responsive small mb-2">
      <table class="table table-dark table-borderless align-middle mb-0">
        <thead class="text-muted text-uppercase small">
          <tr>
            <th>Product</th>
            <th class="text-end">Loaded</th>
            <th class="text-end">Returned</th>
            <th class="text-end">Sold (Loaded - Returned)</th>
            <th class="text-end">Sold (Van Sales)</th>
            <th class="text-end">Diff (Qty)</th>
          </tr>
        </thead>
        <tbody>
          <?php
          $grand_diff = 0;
          foreach ($items as $it):
            $loaded   = (int)$it['qty_loaded'];
            $returned = (int)$it['qty_returned'];
            $sold_sys = $loaded - $returned;

            $vsSoldQty = $vanSalesAgg[$it['product_id']]['sold_qty']   ?? 0;
            $vsSoldVal = $vanSalesAgg[$it['product_id']]['sold_value'] ?? 0.0;

            $diff_qty = $sold_sys - $vsSoldQty;
            $grand_diff += $diff_qty;
          ?>
            <tr>
              <td>
                <strong><?php echo h($it['code']); ?></strong><br>
                <span class="text-muted"><?php echo h($it['name']); ?></span>
              </td>
              <td class="text-end"><?php echo number_format($loaded); ?></td>
              <td class="text-end">
                <?php if ($editable): ?>
                  <input type="hidden" name="dli_id[]" value="<?php echo (int)$it['id']; ?>">
                  <input type="number" name="qty_returned[]" class="form-control form-control-sm text-end"
                         value="<?php echo h($returned); ?>" min="0" max="<?php echo $loaded; ?>">
                <?php else: ?>
                  <?php echo number_format($returned); ?>
                <?php endif; ?>
              </td>
              <td class="text-end"><?php echo number_format($sold_sys); ?></td>
              <td class="text-end">
                <?php echo number_format($vsSoldQty); ?>
              </td>
              <td class="text-end <?php echo $diff_qty == 0 ? 'text-success' : 'text-warning'; ?>">
                <?php echo number_format($diff_qty); ?>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (empty($items)): ?>
            <tr><td colspan="6" class="text-center text-muted py-3">No items on this loading.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($editable && !empty($items)): ?>
      <div class="d-flex justify-content-between align-items-center mt-3">
        <div class="small text-muted">
          Overall diff (system sold vs van sales sold) is currently 
          <strong><?php echo number_format($grand_diff); ?> pcs</strong>.
        </div>
        <button type="submit" class="btn btn-sm btn-primary rounded-pill"
                style="background:var(--sb-primary);border-color:var(--sb-primary-dark);">
          <i class="fa-solid fa-floppy-disk me-1"></i> Save Returns &amp; Reconcile
        </button>
      </div>
    <?php endif; ?>
  </form>
</div>
<?php require_once __DIR__ . "/../../includes/footer.php"; ?>
