<?php
require_once __DIR__ . "/../../config/auth.php";
require_once __DIR__ . "/../../config/db.php";
require_once __DIR__ . "/../../includes/helpers.php";

$id = (int)($_GET['id'] ?? 0);
if (!$id) {
    flash_set('error', 'Missing loading id.');
    redirect('loadings_list.php');
}

// 1) Load header
$stmt = $conn->prepare("SELECT * FROM dispatch_loads WHERE id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();
$load = $stmt->get_result()->fetch_assoc();

if (!$load) {
    flash_set('error', 'Loading not found.');
    redirect('loadings_list.php');
}

if ($load['status'] === 'Posted') {
    flash_set('error', 'This loading is already posted.');
    redirect('loadings_list.php');
}

// 2) Load items
$stmt = $conn->prepare("
    SELECT dli.*, p.code, p.name 
    FROM dispatch_load_items dli
    JOIN products p ON dli.product_id = p.id
    WHERE dli.load_id = ?
");
$stmt->bind_param("i", $id);
$stmt->execute();
$res = $stmt->get_result();
$items = $res->fetch_all(MYSQLI_ASSOC);

if (empty($items)) {
    flash_set('error', 'No items on this loading.');
    redirect('loading_form.php?id='.$id);
}

// 3) Start transaction
$conn->begin_transaction();

try {
    $now = date('Y-m-d H:i:s');

    // First pass: check stock for all items
    foreach ($items as $it) {
        $product_id = (int)$it['product_id'];
        $qty_needed = (int)$it['qty_loaded'];

        // lock row for update
        $stmtStock = $conn->prepare("
            SELECT on_hand 
            FROM finished_goods_stock 
            WHERE product_id = ? 
            FOR UPDATE
        ");
        $stmtStock->bind_param("i", $product_id);
        $stmtStock->execute();
        $stockRow = $stmtStock->get_result()->fetch_assoc();

        $on_hand = $stockRow ? (int)$stockRow['on_hand'] : 0;

        if ($on_hand < $qty_needed) {
            $conn->rollback();
            flash_set('error', "Not enough stock for {$it['code']} ({$it['name']}). On hand: {$on_hand}, needed: {$qty_needed}.");
            redirect('loading_form.php?id='.$id);
        }
    }

    // Second pass: deduct stock + log moves
    foreach ($items as $it) {
        $product_id = (int)$it['product_id'];
        $qty_needed = (int)$it['qty_loaded'];

        // update finished_goods_stock
        $stmtUpd = $conn->prepare("
            UPDATE finished_goods_stock 
            SET on_hand = on_hand - ? 
            WHERE product_id = ?
        ");
        $stmtUpd->bind_param("ii", $qty_needed, $product_id);
        $stmtUpd->execute();

        // insert finished_goods_moves
        $stmtMove = $conn->prepare("
            INSERT INTO finished_goods_moves 
            (move_date, product_id, qty, direction, ref_type, ref_id, remarks)
            VALUES (?,?,?,?,?,?,?)
        ");
        $direction = -1; // out
        $ref_type  = 'DISPATCH';
        $remarks   = 'Loading ' . $load['load_no'] . ' - Route ' . ($load['route_code'] ?? '');
        $stmtMove->bind_param("siiisis", $now, $product_id, $qty_needed, $direction, $ref_type, $id, $remarks);
        $stmtMove->execute();
    }

    // 4) Mark loading as Posted
    $stmt = $conn->prepare("UPDATE dispatch_loads SET status = 'Posted' WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();

    $conn->commit();
    flash_set('success', 'Loading posted and finished goods deducted.');
} catch (Exception $e) {
    $conn->rollback();
    flash_set('error', 'Error posting loading: ' . $e->getMessage());
}

redirect('loadings_list.php');
