<?php
declare(strict_types=1);

function start_session(): void {
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
}

function csrf_token(): string {
  start_session();
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(32));
  return $_SESSION['csrf'];
}

function csrf_check(): void {
  start_session();
  if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $t = (string)($_POST['csrf'] ?? '');
    if (!$t || empty($_SESSION['csrf']) || !hash_equals($_SESSION['csrf'], $t)) {
      http_response_code(419);
      exit('CSRF token mismatch.');
    }
  }
}

function require_login(string $redirect = 'login.php'): array {
  start_session();
  if (empty($_SESSION['user'])) {
    header('Location: ' . $redirect);
    exit;
  }
  return (array)$_SESSION['user'];
}

function has_role(array $u, array $roles): bool {
  $r = (string)($u['role'] ?? '');
  return in_array($r, $roles, true);
}

function login(string $email, string $password): bool {
  $pdo = db();
  $stmt = $pdo->prepare("SELECT id, name, email, role, password_hash, is_active FROM users WHERE email=? LIMIT 1");
  $stmt->execute([$email]);
  $u = $stmt->fetch();
  if (!$u || !(int)$u['is_active']) return false;
  if (!password_verify($password, (string)$u['password_hash'])) return false;

  start_session();
  unset($u['password_hash']);
  $_SESSION['user'] = $u;
  return true;
}

function logout(): void {
  start_session();
  $_SESSION = [];
  if (ini_get('session.use_cookies')) {
    $p = session_get_cookie_params();
    setcookie(session_name(), '', time()-42000, $p['path'], $p['domain'], $p['secure'], $p['httponly']);
  }
  session_destroy();
}
